/*===========================================================================



LX.1407 Improvement SBM-20 based Geiger counter
===============================================



Overview:
=========

CPU         :   Arduino Nano / ATmega328
Clock       :   XTAL 16MHz
Compiler    :   Arduino 1.0.5
Version     :   1.01
Date        :   2017-02-02
Author      :   Iacopo Giangrandi HB9DUL



Description:
============

This program implements the counting functions and unit conversion of as 
SBM-20 based Geiger counter. It accepts pulses on D2 digital input. Only 
this pin (or D3) can be used, because it directly generates interrupt on 
change INT0 (or INT1). Every time a pulse (falling edge) is detected, the 
counter is incremented by one. 

Results are displayed on an 4 digit 7 segment LCD screen and the user can 
use two buttons and a switch to control the unit. The switch has three 
positions, is connected to D16/A2 and D17/A3 and allows selecting between 
mR/h, uSv/h and cpm. Holding button P2 (connected to D18/A4) shows the 
actual number of pulses counted in the current period. Holding button P1 
(connected to D19/A5) will show how many seconds are left until the end of 
the current integration period. Pressing and holding both buttons together 
will show the battery voltage. Battery voltage is sampled via a voltage 
divider on pin D14/A0. 

The LCD display is controlled via four 4094 shift registers which are updated 
100 times per second (generating a 50Hz AC square wave).

Integration period is fixed and set to 10 seconds. It can be changed by 
pressing and holding P1 or P2 while switching the power on until the new time 
is displayed on the screen. P1 will set the integration time to 60 seconds 
and P2 to 2 seconds. 

Each time an integration period ends, the display is updated and the raw 
measurements are also echoed on the serial port at 9600bps. No calculation 
in mR/h nor uSv/h is done on the serial communication, as this can easily 
be done by the remote PC, based on the raw data. Tube conversion constants 
and data format is transmitted at boot time. As a reminder, the SBM-20 
tube produces 175cpm at 1uSv/h and 1530cpm at 1mR/h. Dead time is 190us at 
400V.

If the battery voltage drops below a given thresold (4.5V) a "bAtt" message 
flashes on the screen.



I/O connections:
================

LCD connections:
----------------
D4     ----> LCD_DATA_PIN       LCD serial data (for the 4094 shift register)
D5     ----> LCD_CLOCK_PIN      LCD serial clock (for the 4094 shift register)
D6     ----> LCD_LATCH_PIN      LCD latch (for the 4094 shift register)
D7     ----> LCD_BACK_PLANE     LCD back plane (has to be inverted at 30..200Hz)

UART connections:
-----------------
D0/RXD <---- USB_TX             UART communication (from PC to Arduino)
D1/TXD ----> USB_RX             UART communication (from Arduino to PC)

I/O connections:
----------------
D2     <---- IN_TUBE            Pulses input from the tube
D14/A0 <---- Battery voltage (analog)
D16/A2 <---- IN_MR_H            Short to ground to display mR/h
D17/A3 <---- IN_USV_H           Short to ground to display uSv/h
D18/A4 <---- IN_P2 (COUNT)      Short to ground to display current count
D19/A5 <---- IN_P1 (TIME)       Short to ground to display remaining time



============================================================================= */



// --------------------------------------------------------------------------
// External files inclusions.
// --------------------------------------------------------------------------
#include <TimerOne.h>                                           // Include TMR1 library. Used to drive the LCD at 50Hz.



// --------------------------------------------------------------------------
// General definitions, constants and global variables.
// --------------------------------------------------------------------------
unsigned long int count_tube = 0;                               // This variable is used to count pulses from the tube. It's incremented by interrupt procedure int_tube().
unsigned long int next_millis;                                  // This variable is used to calculate the expiration time of the current integration period.
long int cpm_current = 0;                                       // This variable contains the CPM (counts per minute) of the last completed (current) measurement.

#define   TUBE_R_FACTOR     1530                                // 1530 cpm are equivalent to 1mR/h (SBM-20 specific data).
#define   TUBE_SV_FACTOR    175                                 // 175 cpm are equivalent to 1uSv/h (SBM-20 specific data).
#define   TUBE_DEAD_TIME    190                                 // The SBM-20 tube has a dead time of 190us (SBM-20 specific data).
#define   TIME_SLOW         60000                               // Long integration time is set to 60 seconds (60000ms).
#define   TIME_MED          10000                               // Medium integration time set to 10 seconds (10000ms).
#define   TIME_FAST         2000                                // Short long integration time set to 2 seconds (2000ms).
unsigned long time_interval = TIME_MED;                         // This variable defnes the current integration time. This value will be overwritten anyway by set_integration_time() called in setup().

#define   SEVEN_SEG_0       B11111100                           // These constants implement the symbols on the 7-segment displays.
#define   SEVEN_SEG_1       B01100000                           // In the order, each bit represents  the segments: a, b, c, d, e, f, g, dp.
#define   SEVEN_SEG_2       B11011010                           // And 1 equals "on".
#define   SEVEN_SEG_3       B11110010
#define   SEVEN_SEG_4       B01100110
#define   SEVEN_SEG_5       B10110110
#define   SEVEN_SEG_6       B10111110
#define   SEVEN_SEG_7       B11100100
#define   SEVEN_SEG_8       B11111110
#define   SEVEN_SEG_9       B11100110

#define   SEVEN_SEG_A       B11101110                           // A
#define   SEVEN_SEG_B       B00111110                           // b
#define   SEVEN_SEG_C       B10011100                           // C
#define   SEVEN_SEG_D       B01111010                           // d
#define   SEVEN_SEG_E       B10011110                           // E
#define   SEVEN_SEG_H       B01101110                           // H
#define   SEVEN_SEG_I       B01100000                           // I
#define   SEVEN_SEG_U       B01111100                           // U
#define   SEVEN_SEG_L       B00011100                           // L
#define   SEVEN_SEG_M       B11101100                           // M
#define   SEVEN_SEG_O       B11111100                           // O
#define   SEVEN_SEG_R       B00001010                           // r
#define   SEVEN_SEG_S       B10110110                           // S
#define   SEVEN_SEG_T       B00011110                           // t
#define   SEVEN_SEG_V       B00111000                           // v

#define   SEVEN_SEG_DASH    B00000010                           // "-"
#define   SEVEN_SEG_DOT     B00000001                           // Decimal point, just "or" it to the symbol to switch it on.
#define   SEVEN_SEG_BLANK   B00000000                           // " "

byte digits[] =                                                 // An array of digits used to convert numbers.
{
    SEVEN_SEG_0,
    SEVEN_SEG_1,
    SEVEN_SEG_2,
    SEVEN_SEG_3,
    SEVEN_SEG_4,
    SEVEN_SEG_5,
    SEVEN_SEG_6,
    SEVEN_SEG_7,
    SEVEN_SEG_8,
    SEVEN_SEG_9
};

#define   DISPLAY_SIZE      4                                   // Number of digits on the display (4).

unsigned char lcd_display[DISPLAY_SIZE] =                       // This array is the what actually is displayed. Just change the content of it to display it. It's read by the ISR 100 times a second. Start with all segments on.
{
    SEVEN_SEG_8 | SEVEN_SEG_DOT, 
    SEVEN_SEG_8 | SEVEN_SEG_DOT, 
    SEVEN_SEG_8 | SEVEN_SEG_DOT, 
    SEVEN_SEG_8 | SEVEN_SEG_DOT 
};

#define   LATCH_PIN         5                                   // Pin connected to STR (pin 1) of HCF4094
#define   CLOCK_PIN         6                                   // Pin connected to CP (pin 3) of HCF4094
#define   DATA_PIN          4                                   // Pin connected to D (pin 2) of HCF4094
#define   BACK_PLANE_PIN    7

#define   IN_TUBE           2                                   // Input for pulses of the tube. Must be on D2 to use INT0. Triggers on falling edge.
#define   IN_MR_H           16                                  // Input to select mR/h (LOW=selected). Uses internal pull-up.
#define   IN_USV_H          17                                  // Input to select uS/h (LOW=selected). Uses internal pull-up.
#define   IN_P1             19                                  // Input for button P1 (UNIT). LOW=pressed. Uses internal pull-up.
#define   IN_P2             18                                  // Input for button P2 (MODE). LOW=pressed. Uses internal pull-up.
#define   VOLT_BATT         14                                  // Analog input for battery rail (through voltage divider).

#define   VBATT_MIN         4500                                // Alkaline batteries are conidered low if below 4.5V

#define   CYCLE_DELAY       100                                 // Time to wait at the end of each cycle, 100ms.



// --------------------------------------------------------------------------
// This is the setup procedure executed once at startup. This procedure is
// automatically executed by Arduino.
// --------------------------------------------------------------------------
void setup(void) 
{

    // Configure I/O lines.

    pinMode(IN_TUBE, INPUT);                                    // Configure tube pulse pin as input.
    pinMode(IN_P1, INPUT_PULLUP);                               // Configure buttons and jumpers as inputs with pullup resistor.
    pinMode(IN_P2, INPUT_PULLUP);
    pinMode(IN_MR_H, INPUT_PULLUP);
    pinMode(IN_USV_H, INPUT_PULLUP);
    attachInterrupt(0, int_tube, FALLING);                      // Enable interrupt on change for D2.

    analogReference(INTERNAL);                                  // Use the internal 1.1V analog reference.
    pinMode(VOLT_BATT, INPUT);                                  // Voltage input on A0(D14) to read battery voltage.

    // Configure and initialize LCD display.

    pinMode(LATCH_PIN, OUTPUT);                                 // LCD pins are outputs.
    pinMode(CLOCK_PIN, OUTPUT);
    pinMode(DATA_PIN, OUTPUT);
    pinMode(BACK_PLANE_PIN, OUTPUT);    
    Timer1.initialize(10000);                                   // Set timer 1 every 10000us, so that the ISR will be called 100 times a second.
    Timer1.attachInterrupt(int_timer);                          // Attach the service routine here.

    // Configure serial port.

    Serial.begin(9600);                                         // Set serial communication at 9600N81.

    // Now we are done with initializations, we can show the welocme message.

    display_welcome_screen();                                   // Start with the usual greetings.
    set_integration_time();                                     // Check if P1 or P2 are pressed to modify the integration time.
    
    next_millis = millis() + time_interval;                     // Set duration for first measurement.
}



// --------------------------------------------------------------------------
// This is the main program. It's executed automatically after the 
// initialization in an andless loop by Arduino.
// --------------------------------------------------------------------------
void loop(void)
{
    long int count_tube_tmp;                                    // Temporary variables to store current readings.
    int volt_batt;

    // Measure battery voltage and dose.

    volt_batt = read_voltage(VOLT_BATT);                        // Read battery voltage.

    if (millis() >= next_millis)                                // Check if integration time is elapsed.
    {
        next_millis = millis() + time_interval;                 // Yes, measurement is done: compute next interval.
        count_tube_tmp = count_tube;
        count_tube = 0;                                         // Reset real-time counter.

        cpm_current = count_tube_tmp;                           // Compute counts per minute.
        cpm_current *= 60;
        cpm_current /= (time_interval / 1000);
        cpm_current = compensate_dead_time(cpm_current);        // Compensate for tube dead time.

        Serial.print(time_interval);                            // Communicate result on the serial port.
        Serial.print(';');
        Serial.print(count_tube_tmp);
        Serial.print(';');
        Serial.print(cpm_current);
        Serial.print(';');
        Serial.println(volt_batt);
    }

    // Display the result or the current status.

    if (digitalRead(IN_P1) == LOW)                              // First check pushbuttons
    {
        if (digitalRead(IN_P2) == LOW)                          // Both buttons pressed: display voltage
        {
            display_value_divided_by_100(volt_batt / 10);
        }
        else                                                    // Only P1 is pressed: display time left
        {
            display_value_divided_by_10(long(next_millis - millis()) / 100);
        }
    }
    else if (digitalRead(IN_P2) == LOW)                         // Only P2 is pressed: display current count
    {
        display_value(count_tube);
    }
    else                                                        // No button is pressed: display dose according to the unit switch
    {
        if (digitalRead(IN_MR_H) == LOW)                        // Display mR/h
        {
            unsigned long int micro_roentgen_hour = cpm_current * 1000;
            micro_roentgen_hour /= TUBE_R_FACTOR;
            display_value_divided_by_1000(micro_roentgen_hour);
        }
        else if (digitalRead(IN_USV_H) == LOW)                  // Display uSv/h
        {
            unsigned long int nano_sievert_hour = cpm_current * 1000;
            nano_sievert_hour /= TUBE_SV_FACTOR;
            display_value_divided_by_1000(nano_sievert_hour);
        }
        else                                                    // Display CPM
        {
            display_value(cpm_current);
        }
    }

    check_battery(volt_batt);

    delay(CYCLE_DELAY);                                         // Wait some time and cycle again (and again).
}



// --------------------------------------------------------------------------
// This procedure compensate for the tube dead time. When a radioactive 
// particle strikes the tube, the gas is ionized and a pulse is generated. 
// Then the ionization disappears and the tube is ready again for the next 
// particle, but as long as it’s ionized, it cannot detect another particle. 
// This is called dead time and de facto shortens the integration time. If 
// "f" is the measured pulse frequency and "tau" is the tube dead time, the 
// corrected pulse frequency "f'" can be expressed as follows:
//
// f' = f / (1 - f * tau)
//
// But time units for "f" and "tau" must match. Here "f" is in cpm and "tau" 
// (TUBE_DEAD_TIME) in microseconds, so some conversion has to take place. 
// Furthermore, in order to use integer math functions, all values are 
// multiplied by 1000. 
// The SMB-20 has a dead time of 190us and a maximum frequency of 220'000cpm, 
// so with a factor of 1000, we get sufficient accuracy while still fitting 
// into long int varaibles.
// The correction increases as the count increases. For the SBM-20 tube, the 
// correction is about 1% at 3'000cpm, rises at 10% at 30'000cpm and is 50% 
// at 160'000cpm.
// --------------------------------------------------------------------------
long int compensate_dead_time(long int measured_cpm)
{
    measured_cpm = 1000L * measured_cpm / (1000L - measured_cpm * TUBE_DEAD_TIME / (1000L * 60L));
    return (measured_cpm);
}



// --------------------------------------------------------------------------
// Interrupt service routine (INT0) run each time the tube generates a pulse.
// --------------------------------------------------------------------------
void int_tube(void)
{
    count_tube++;
}



// --------------------------------------------------------------------------
// This procedure is executed by TMR1 every 10ms (100 times per second). 
// Every time it's called, the back plane pin is inverted, de facto driving 
// the LCD screen at half this frequency (50Hz). Every time is called, the 
// display variable lcd_display[] is read and transferred to the four 4094 
// shift registers to directly drive the segments. Of course, when the 
// backplane is inverted, the data in the shift register is inverted as 
// well.
// --------------------------------------------------------------------------
void int_timer(void)
{
    int i;                                                      // A counter to parse through the 4 digits.
    if (digitalRead(BACK_PLANE_PIN) == LOW)                     // Check the status of the back plane: is it low?
    {
        for (i = 0; i < DISPLAY_SIZE; i++)                      // Yes, parse all four digits.
        {
            digitalWrite(LATCH_PIN, HIGH);                      // Rise the latch pin before transfering.
            shiftOut(DATA_PIN, CLOCK_PIN, LSBFIRST, ~lcd_display[i]);   // Transfet the four digits inverted because the backplane will go high.
            digitalWrite(LATCH_PIN, LOW);                       // Drop the latch pin to validate the data.
        }
        digitalWrite(BACK_PLANE_PIN, HIGH);                     // Now set the back plane high.
    }
    else                                                        // No, back plane is high and will go low.
    {
        for (i = 0; i < DISPLAY_SIZE; i++)                      // Parse all four digits.
        {
            digitalWrite(LATCH_PIN, HIGH);                      // Rise the latch pin before transfering.
            shiftOut(DATA_PIN, CLOCK_PIN, LSBFIRST, lcd_display[i]);    // Transfet the four digits normally because the backplane will go low.
            digitalWrite(LATCH_PIN, LOW);                       // Drop the latch pin to validate the data.
        }
        digitalWrite(BACK_PLANE_PIN, LOW);                      // Now clear the back plane high.
    }
}



// --------------------------------------------------------------------------
// This procedure will display the specified value on the four digits 
// display. It will appear divided by 1000: if you set value to 123, the 
// display will show "0.123". For values larger than 10000, the decimal point 
// is shifted accordingly: 15678 will be shown as "15.67". Values larger than 
// 100000 will be shown divided by 1000000 and a little triangle will tell 
// the user about the extra 1000 factor. So a value of 1234567 will be 
// displayed as "^1.234". Negative values are shown as "----".
// --------------------------------------------------------------------------
void display_value_divided_by_1000(long int value)
{
    int i;
    byte digit;
    int multiplier = 0;

    if (value < 0)                                              // Negative values will be displayed as "----"
    {
        lcd_display[3] = SEVEN_SEG_DASH;
        lcd_display[2] = SEVEN_SEG_DASH;
        lcd_display[1] = SEVEN_SEG_DASH;
        lcd_display[0] = SEVEN_SEG_DASH;
        return;
    }

    while (value > 9999)                                        // Make sure the value will fit in the display by dividing by 10.
    {
        value /= 10;
        multiplier++;                                           // Count how many times we divided by 10.
    }

    for (i=0; i<4; i++)                                         // Convert the integer into characters.
    {
        digit = value % 10;
        lcd_display[i] = digits[digit];
        value /= 10;
    }
    
    switch (multiplier)                                         // Place the decimal point accordingly.
    {
        case 0: case 3: lcd_display[3] |= SEVEN_SEG_DOT; break;
        case 1: case 4: lcd_display[2] |= SEVEN_SEG_DOT; break;
        case 2: case 5: lcd_display[1] |= SEVEN_SEG_DOT; break;
    }
    if (multiplier >= 3)                                        // If we divided by more than 1000, set the little triangle on the display to say that the result is 1000 times smaller.
        lcd_display[0] |= SEVEN_SEG_DOT;
}



// --------------------------------------------------------------------------
// This procedure will display the specified value on the four digits 
// display. Leading zeros are suppressed: 123 is displayed as " 123". For 
// values larger than 10000, the decimal point is shifted accordingly and a 
// little triangle will tell the user about the extra 1000 factor. 15678 
// will be shown as "^15.67". Negative values are shown as "----", values 
// larger than 9'999'999 are displayed as "EEEE".
// --------------------------------------------------------------------------
void display_value(long int value)
{
    int i;
    byte digit;
    int multiplier = 0;

    if (value < 0)                                              // Negative values will be displayed as "----"
    {
        lcd_display[3] = SEVEN_SEG_DASH;
        lcd_display[2] = SEVEN_SEG_DASH;
        lcd_display[1] = SEVEN_SEG_DASH;
        lcd_display[0] = SEVEN_SEG_DASH;
        return;
    }
    if (value > 9999999)                                        // Over range values are displayed as "EEEE"
    {
        lcd_display[3] = SEVEN_SEG_E;
        lcd_display[2] = SEVEN_SEG_E;
        lcd_display[1] = SEVEN_SEG_E;
        lcd_display[0] = SEVEN_SEG_E;
        return;
    }

    while (value > 9999)                                        // Make sure the value will fit in the display by dividing by 10.
    {
        value /= 10;
        multiplier++;                                           // Count how many times we divided by 10.
    }

    for (i=0; i<4; i++)                                         // Convert the integer into characters.
    {
        digit = value % 10;
        lcd_display[i] = digits[digit];
        value /= 10;
    }

    if (multiplier == 0)                                        // Remove leading zeros only if we didn't divide by 10.
    {
        i = 3;
        while (lcd_display[i] == SEVEN_SEG_0)                   // Parse all digits, left to right.
        {
            lcd_display[i] = SEVEN_SEG_BLANK;                   // And replace zeros with spaces.
            i--;
            if (i == 0)                                         // Leave the rightmost zero.
                break;
        }
    }

    switch (multiplier)                                         // Place the decimal point accordingly.
    {
        case 1: lcd_display[2] |= SEVEN_SEG_DOT; break;
        case 2: lcd_display[1] |= SEVEN_SEG_DOT; break;
    }
    if (multiplier >= 1)                                        // If we divided by 10 at least once, set the little triangle on the display to say that the result is 1000 times smaller.
        lcd_display[0] |= SEVEN_SEG_DOT;
}



// --------------------------------------------------------------------------
// This procedure will display the specified value on the four digits 
// display divided by 10. Leading zeros are suppressed: 123 is displayed as 
// " 12.3". Negative values are shown as "----", values larger than 9'999 
// are displayed as "EEEE".
// --------------------------------------------------------------------------
void display_value_divided_by_10(int value)
{
    int i;
    byte digit;

    if (value < 0)                                              // Negative values will be displayed as "----"
    {
        lcd_display[3] = SEVEN_SEG_DASH;
        lcd_display[2] = SEVEN_SEG_DASH;
        lcd_display[1] = SEVEN_SEG_DASH;
        lcd_display[0] = SEVEN_SEG_DASH;
        return;
    }
    if (value > 9999)                                           // Over range values are displayed as "EEEE"
    {
        lcd_display[3] = SEVEN_SEG_E;
        lcd_display[2] = SEVEN_SEG_E;
        lcd_display[1] = SEVEN_SEG_E;
        lcd_display[0] = SEVEN_SEG_E;
        return;
    }

    for (i=0; i<4; i++)                                         // Convert the integer into characters.
    {
        digit = value % 10;
        lcd_display[i] = digits[digit];
        value /= 10;
    }

    i = 3;                                                      // Remove leading zeros
    while (lcd_display[i] == SEVEN_SEG_0)                       // Parse all digits, left to right.
    {
        lcd_display[i] = SEVEN_SEG_BLANK;                       // And replace zeros with spaces.
        i--;
        if (i <= 1)                                             // Leave two zeros on the right.
            break;
    }

    lcd_display[1] |= SEVEN_SEG_DOT;                            // Set decimal point at "XXX.X".
}



// --------------------------------------------------------------------------
// This procedure will display the specified value on the four digits 
// display divided by 100. Leading zero is suppressed: 123 is displayed as 
// " 1.23". Negative values are shown as "----", values larger than 9'999 
// are displayed as "EEEE".
// --------------------------------------------------------------------------
void display_value_divided_by_100(int value)
{
    int i;
    byte digit;

    if (value < 0)                                              // Negative values will be displayed as "----"
    {
        lcd_display[3] = SEVEN_SEG_DASH;
        lcd_display[2] = SEVEN_SEG_DASH;
        lcd_display[1] = SEVEN_SEG_DASH;
        lcd_display[0] = SEVEN_SEG_DASH;
        return;
    }
    if (value > 9999)                                           // Over range values are displayed as "EEEE"
    {
        lcd_display[3] = SEVEN_SEG_E;
        lcd_display[2] = SEVEN_SEG_E;
        lcd_display[1] = SEVEN_SEG_E;
        lcd_display[0] = SEVEN_SEG_E;
        return;
    }

    for (i=0; i<4; i++)                                         // Convert the integer into characters.
    {
        digit = value % 10;
        lcd_display[i] = digits[digit];
        value /= 10;
    }

    if (lcd_display[3] == SEVEN_SEG_0)                          // Remove leading zero
        lcd_display[3] = SEVEN_SEG_BLANK;

    lcd_display[2] |= SEVEN_SEG_DOT;                            // Set decimal point at "XX.XX".
}



// --------------------------------------------------------------------------
// This procedure will disply the welcome screen on the LCD display and also
// send it on the serial port.
// --------------------------------------------------------------------------
void display_welcome_screen(void)
{
    delay(500);

    lcd_display[3] = SEVEN_SEG_H;                               // Display "Hb9 "
    lcd_display[2] = SEVEN_SEG_B;
    lcd_display[1] = SEVEN_SEG_9;
    lcd_display[0] = SEVEN_SEG_BLANK;

    Serial.println("LX.1407 Geiger counter");                   // Print welcome msage also on the serial port.
    Serial.println("Iacopo Giangrandi - 2017 - v1.0");
    Serial.println();

    delay(500);

    lcd_display[3] = SEVEN_SEG_BLANK;                           // Display " dUL"
    lcd_display[2] = SEVEN_SEG_D;
    lcd_display[1] = SEVEN_SEG_U;
    lcd_display[0] = SEVEN_SEG_L;

    Serial.print("T_off = ");                                   // Print tube constants on the serial port.
    Serial.print(TUBE_DEAD_TIME);
    Serial.print(" us");
    Serial.println();
    Serial.print("1 mR/h = ");
    Serial.print(TUBE_R_FACTOR);
    Serial.println(" cpm");
    Serial.print("1 uSv/h = ");
    Serial.print(TUBE_SV_FACTOR);
    Serial.println(" cpm");
    Serial.println();

    delay(500);

    Serial.println("Time period [ms];Count tube [1];Calculated counts per minute [cpm];Battery voltage [mV]");  // Print data format on the serial port.
}



// --------------------------------------------------------------------------
// This procedure will check if the buttons are held down and set and 
// display the integration time accordingly. This procedure is called at 
// initialization time.
// --------------------------------------------------------------------------
void set_integration_time(void)
{
    if (digitalRead(IN_P1) == LOW)                              // Check if P1 is pressed.
    {
        time_interval = TIME_SLOW;                              // Yes: set long integration time.
        display_value_divided_by_10(time_interval / 100);       // Display it.
        while (digitalRead(IN_P1) == LOW) ;                     // And wait until the button is released.
    }
    else if (digitalRead(IN_P2) == LOW)                         // Check if P2 is pressed.
    {
        time_interval = TIME_FAST;                              // Yes: set short integration time.
        display_value_divided_by_10(time_interval / 100);       // Display it.
        while (digitalRead(IN_P2) == LOW) ;                     // And wait until the button is released.
    }
    else                                                        // No button is pressed.
    {
        time_interval = TIME_MED;                               // Set medium integration time.
        display_value_divided_by_10(time_interval / 100);       // Display it.
        delay(500);                                             // And wait half a second.
    }
}



// --------------------------------------------------------------------------
// This procedure will return the rail voltage by reading the stepped down 
// voltage on the specified analog pin.
// The voltage divider is composed by two resistors R1=22k and R2=3k3, step 
// down factor is therefore 1:7.7. 
// The ADC converter in the Arduino has a resolution of ADC_M=1024 points 
// and a voltage reference U_ref=1.1V. Knowing the ADC raw reading ADC#, we 
// can calculate the rail voltage U_rail with the following relation:
//
// U_rail = ADC# * (U_ref / ADC_M) * ((R1 + R2) / R2)
//
// Voltage is returned in millivolts, but the resolution is only 
//
// resolution = (U_ref / ADC_M) * ((R1 + R2) / R2) = 8.2 mV
//
// It's important to use unsigned long int variables and specify constants 
// as "UL" to avoid rounding errors.
// --------------------------------------------------------------------------
unsigned int read_voltage(char pin)
{
    unsigned long adc_value;

    adc_value = analogRead(pin);                                // Read the voltage on VOLT_IN.

    adc_value = adc_value * 1100UL * (33UL + 220UL);             // Calculate rail voltage with the equation explained above.
    adc_value = adc_value / (1024UL * 33UL);
    return(adc_value);
}



// --------------------------------------------------------------------------
// This procedure will compare the passed voltage and display a low battery 
// message if the battery is low. Every 40 time this procedure is called, the
// message is displayed only for the five first times, so that it blinks 
// between the normal readings, and the instrument can still be used.
// --------------------------------------------------------------------------
void check_battery(int vbatt)
{
    static byte blink_counter = 0;                              // Counter used to slow down the blinking of the empty battery.

    if (vbatt < VBATT_MIN)                                      // Check voltage.
    {
        if (blink_counter < 5)                                  // Cycle between the two empty battery symbols.
        {
            lcd_display[3] = SEVEN_SEG_B;
            lcd_display[2] = SEVEN_SEG_A;
            lcd_display[1] = SEVEN_SEG_T;
            lcd_display[0] = SEVEN_SEG_T;
        }
        blink_counter++;
        if (blink_counter > 40)
            blink_counter = 0;
    }
}



